package com.saas.admin.entity;

import jakarta.persistence.*;
import lombok.*;
import org.hibernate.annotations.JdbcTypeCode;
import org.hibernate.type.SqlTypes;

import java.time.LocalDateTime;
import java.util.List;

/**
 * Admin-level inbound call data entity
 * Stored in saas_db for centralized cross-tenant call tracking and analytics
 */
@Entity
@Table(name = "inbound_call_data")
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class AdminInboundCallData {
    
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private Long id;
    
    @Column(name = "tenant_id", nullable = false, length = 100)
    private String tenantId;
    
    // Call identifiers
    @Column(name = "call_sid", nullable = false, unique = true)
    private String callSid;
    
    @Column(name = "parent_call_sid")
    private String parentCallSid;
    
    // Call participants
    @Column(name = "from_number", length = 50)
    private String fromNumber;
    
    @Column(name = "to_number", length = 50)
    private String toNumber;
    
    @Column(name = "caller", length = 50)
    private String caller;
    
    @Column(name = "called", length = 50)
    private String called;
    
    @Column(name = "forwarded_from", length = 50)
    private String forwardedFrom;
    
    // Call metadata
    @Column(name = "provider", length = 50)
    private String provider;  // TWILIO, TELNYX, ELEVENLABS, ZIWO
    
    @Column(name = "direction", length = 20)
    private String direction;
    
    @Column(name = "call_status", length = 50)
    private String callStatus;
    
    @Column(name = "call_token")
    private String callToken;
    
    // Geographic data
    @Column(name = "from_city", length = 100)
    private String fromCity;
    
    @Column(name = "from_state", length = 100)
    private String fromState;
    
    @Column(name = "from_zip", length = 20)
    private String fromZip;
    
    @Column(name = "from_country", length = 100)
    private String fromCountry;
    
    @Column(name = "to_city", length = 100)
    private String toCity;
    
    @Column(name = "to_state", length = 100)
    private String toState;
    
    @Column(name = "to_zip", length = 20)
    private String toZip;
    
    @Column(name = "to_country", length = 100)
    private String toCountry;
    
    @Column(name = "caller_city", length = 100)
    private String callerCity;
    
    @Column(name = "caller_state", length = 100)
    private String callerState;
    
    @Column(name = "caller_zip", length = 20)
    private String callerZip;
    
    @Column(name = "caller_country", length = 100)
    private String callerCountry;
    
    @Column(name = "called_city", length = 100)
    private String calledCity;
    
    @Column(name = "called_state", length = 100)
    private String calledState;
    
    @Column(name = "called_zip", length = 20)
    private String calledZip;
    
    @Column(name = "called_country", length = 100)
    private String calledCountry;
    
    // Call timing
    @Column(name = "start_time")
    private LocalDateTime startTime;
    
    @Column(name = "end_time")
    private LocalDateTime endTime;
    
    @Column(name = "duration")
    private Integer duration;  // in seconds
    
    // Twilio/Telnyx specific fields
    @Column(name = "account_sid")
    private String accountSid;
    
    @Column(name = "api_version", length = 50)
    private String apiVersion;
    
    @Column(name = "recording_sid")
    private String recordingSid;
    
    @Column(name = "recording_url", length = 500)
    private String recordingUrl;
    
    @Column(name = "stir_verstat", length = 50)
    private String stirVerstat;
    
    // Conversation transcript (JSON array of messages)
    @JdbcTypeCode(SqlTypes.JSON)
    @Column(name = "conversation", columnDefinition = "JSON")
    private List<Object> conversation;
    
    // Timestamps
    @Column(name = "created_at", nullable = false, updatable = false)
    private LocalDateTime createdAt;
    
    @Column(name = "updated_at")
    private LocalDateTime updatedAt;
    
    @PrePersist
    protected void onCreate() {
        createdAt = LocalDateTime.now();
        updatedAt = LocalDateTime.now();
    }
    
    @PreUpdate
    protected void onUpdate() {
        updatedAt = LocalDateTime.now();
    }
}
