package com.saas.shared.audit;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import jakarta.servlet.http.HttpServletRequest;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

@Service
@Slf4j
@RequiredArgsConstructor
public class AuditService {
    
    private final AuditRepository auditRepository;
    private final ObjectMapper objectMapper;
    
    @Async
    public void log(String userId, String tenantId, String action, String entityType, 
                    String entityId, Object details, HttpServletRequest request) {
        try {
            String detailsJson = details != null ? objectMapper.writeValueAsString(details) : null;
            String ipAddress = getClientIpAddress(request);
            
            AuditLog auditLog = AuditLog.builder()
                .userId(userId)
                .tenantId(tenantId)
                .action(action)
                .entityType(entityType)
                .entityId(entityId)
                .details(detailsJson)
                .ipAddress(ipAddress)
                .build();
            
            auditRepository.save(auditLog);
            log.debug("Audit log saved: {} - {} - {}", userId, action, entityType);
        } catch (JsonProcessingException e) {
            log.error("Failed to serialize audit details", e);
        } catch (Exception e) {
            log.error("Failed to save audit log", e);
        }
    }
    
    private String getClientIpAddress(HttpServletRequest request) {
        if (request == null) {
            return "UNKNOWN";
        }
        
        String xForwardedFor = request.getHeader("X-Forwarded-For");
        if (xForwardedFor != null && !xForwardedFor.isEmpty()) {
            return xForwardedFor.split(",")[0].trim();
        }
        
        String xRealIp = request.getHeader("X-Real-IP");
        if (xRealIp != null && !xRealIp.isEmpty()) {
            return xRealIp;
        }
        
        return request.getRemoteAddr();
    }
}
