<?php

namespace App\Packages\Creatify\Api;

use Illuminate\Http\JsonResponse;

class LinkToVideos
{
    public function __construct(protected BaseApiClient $client) {}

    /**
     * APIs that convert any link to a short form video ad
     *
     * @see https://docs.creatify.ai/api-reference/link_to_videos/post-apilink_to_videos
     */
    public function createVideoFromLink(array $params): JsonResponse
    {
        $res = $this->client->request('post', 'api/link_to_videos/', $params);

        return $this->client->jsonStatusResponse($res);
    }

    /**
     * Get video result
     *
     * @see https://docs.creatify.ai/api-reference/link_to_videos/get-apilink_to_videos-
     */
    public function getVideoResult(string $id): JsonResponse
    {
        $res = $this->client->request('get', "api/link_to_videos/$id/");

        return $this->client->jsonStatusResponse($res);
    }

    /**
     * Retrieve a list of videos, with an optional ids param to filter vdieo by their UUIDs.
     *
     * @see https://docs.creatify.ai/api-reference/link_to_videos/get-apilink_to_videos
     */
    public function getVideoHistory(?string $ids = null): JsonResponse
    {
        $res = $this->client->request('get', 'api/link_to_videos/', $ids ? ['ids' => $ids] : null);

        return $this->client->jsonStatusResponse($res);
    }

    /**
     * This endpoint generates a preiview of video before rendering the final video.
     *
     * @see https://docs.creatify.ai/api-reference/link_to_videos/post-apilink_to_videos_preview
     */
    public function generatePreviewVideoFromLink(array $params): JsonResponse
    {
        $res = $this->client->request('post', 'api/link_to_videos/preview/', $params);

        return $this->client->jsonStatusResponse($res);
    }

    /**
     * This endpoint renders a video with preview genereated.
     *
     * @see https://docs.creatify.ai/api-reference/link_to_videos/post-apilink_to_videos_render
     */
    public function renderVideo(string $id): JsonResponse
    {
        $res = $this->client->request('post', "api/link_to_videos/$id/render/");

        return $this->client->jsonStatusResponse($res);
    }

    /**
     * This endpoint generates a list of preview videos asynchronously before rendering the final video
     *
     * @see https://docs.creatify.ai/api-reference/link_to_videos/post-apilink_to_videos_preview_list_aysnc
     */
    public function generateListOfPreviews(array $params): JsonResponse
    {
        $res = $this->client->request('post', 'api/link_to_videos/preview_list_async/', $params);

        return $this->client->jsonStatusResponse($res);
    }

    /**
     * This endpoint renders a video from the list of previews generated by Generate a list of Previews Async API.
     *
     *
     * @see https://docs.creatify.ai/api-reference/link_to_videos/post-apilink_to_videos_render_single_preview
     *
     * @param  string  $id  //video id to render
     * @param  string  $media_job  //media_job is the unique identifier of a preview video url. You can get it from the previews field.
     */
    public function renderVideoFromListOfPreviews(string $id, string $media_job): JsonResponse
    {
        $res = $this->client->request('post', "api/link_to_videos/$id/render_single_preview/", ['media_job' => $media_job]);

        return $this->client->jsonStatusResponse($res);
    }
}
