<?php

namespace App\Services\Common;

use App\Domains\Engine\Enums\EngineEnum;
use App\Enums\Introduction;
use App\Helpers\Classes\Helper;
use App\Helpers\Classes\Localization;
use App\Helpers\Classes\MarketplaceHelper;
use App\Helpers\Classes\MenuHelper;
use App\Helpers\Classes\PlanHelper;
use App\Models\Common\Menu;
use App\Models\Extension;
use App\Models\Integration\Integration;
use App\Models\Plan;
use App\Models\Setting;
use App\Models\SettingTwo;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;

class MenuService
{
    public const MENU_KEY = 'dynamic_menu_key';

    public function cacheKey(): string
    {
        $userType = Auth::user()?->isAdmin() || Auth::user()?->isSuperAdmin() ? 'admin' : 'user';

        $userPlan = PlanHelper::userPlan();

        $planId = 0;

        if ($userPlan) {
            $planId = $userPlan->getAttribute('id');
        }

        return $this->fullCacheKey($userType, $planId);
    }

    public function fullCacheKey($userType = null, $planId = null): string
    {
        $locale = Localization::getLocale();

        $byUserId = config('app.menu_cache_by_user_id', false);

        if ($byUserId && Auth::check()) {
            $userId = Auth::id();

            return $locale . '-' . $userType . '-' . $planId . '-' . get_theme() . '-navbar-menu-' . $userId;
        }

        return $locale . '-' . $userType . '-' . $planId . '-' . get_theme() . '-navbar-menu';
    }

    public function regenerate(): array
    {
        $this->cacheClearBlade();

        cache()->forget(self::MENU_KEY);

        return $this->generate(false);
    }

    public function cacheClearBlade(): void
    {
        $byUserId = config('app.menu_cache_by_user_id', false);

        if ($byUserId) {
            Artisan::call('optimize:clear');

            return;
        }

        Plan::query()
            ->where('plans.type', '=', 'subscription')
            ->get()
            ?->map(function ($plan) {
                cache()->forget($this->fullCacheKey('admin', $plan->getAttribute('id')));
                cache()->forget($this->fullCacheKey('user', $plan->getAttribute('id')));
            });

        cache()->forget($this->fullCacheKey('admin'));
        cache()->forget($this->fullCacheKey('user'));
    }

    public function boltMenu(): array
    {
        $data = collect($this->generate(false))->where('bolt_menu', true);

        $array = [];

        foreach ($data as $key => $value) {
            $array[$key] = [
                'background' => $value['bolt_background'],
                'foreground' => $value['bolt_foreground'],
            ];
        }

        return $array;
    }

    public function generate(bool $active = true): array
    {

        $data = cache()->rememberForever(self::MENU_KEY, function () use ($active) {
            $items = Menu::query()
                ->with('children')
                ->withCount('children')
                ->whereNull('parent_id')
                ->when($active, function ($query) {
                    $query->where('is_active', true);
                })
                ->orderBy('order', 'asc')
                ->get();

            return $this->merge($items);
        });

        if (setting('dash_theme') === 'oupi') {
            $menuHelper = app(MenuHelper::class);

            foreach ($data as $key => $value) {
                $data[$key]['performUserCheck'] = (bool) $menuHelper->checker($value);
            }
        }

        return $data;
    }

    public function subMenuOrderUpdate(array $data)
    {

        $order = 0;

        $lastParent = 0;

        foreach ($data as $key => $value) {
            if ($value != $lastParent) {
                $order = 0;
            } else {
                $order = $order + 1;
            }

            Menu::query()
                ->where('id', $key)
                ->update([
                    'order'     => $order,
                    'parent_id' => $value,
                ]);

            $lastParent = $value;
        }
    }

    public function parentMenuOrderUpdate(array $data): void
    {
        foreach ($data as $key => $value) {
            Menu::query()
                ->where('id', $value)
                ->update([
                    'parent_id' => null,
                    'order'     => $key,
                ]);
        }
    }

    public function merge($items, $else = true): array
    {
        $data = [];

        $staticData = $this->data();

        foreach ($items as $item) {

            if (isset($staticData[$item['key']])) {

                $data[$item['key']] = array_merge($staticData[$item['key']], $item->toArray());

                if ($item->parent_id == null && $else) {
                    $children = $item->getAttribute('children');

                    $data[$item['key']]['children'] = $this->merge($children);
                }

            } elseif ($else) {
                $data[$item['key']] = array_merge($item->toArray(), [
                    'active_condition' => false,
                    'show_condition'   => true,
                    'extension'        => false,
                ]);
                if ($item->parent_id == null && $else) {
                    $children = $item->getAttribute('children');

                    $data[$item['key']]['children'] = $this->merge($children);
                }
            }
        }

        return $data;
    }

    public function data(): array
    {
        $admin = Auth::user()?->isAdmin();

        $setting = Setting::getCache();
        $settings_two = SettingTwo::getCache();

        $menu = [

            'user_label' => [
                'parent_key'       => null,
                'key'              => 'user_label',
                'route'            => null,
                'label'            => 'User',
                'icon'             => null,
                'svg'              => null,
                'order'            => 1,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'label',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
            ],
            'dashboard' => [
                'parent_key'       => null,
                'key'              => 'dashboard',
                'route'            => 'dashboard.user.index',
                'label'            => 'Dashboard',
                'icon'             => 'tabler-layout-2',
                'svg'              => null,
                'order'            => 2,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.index',
                ],
                'show_condition' => true,
            ],
            'creative_suite' => [
                'parent_key'       => null,
                'key'              => 'creative_suite',
                'route'            => 'dashboard.user.creative-suite.index',
                'label'            => 'Creative Suite',
                'icon'             => 'tabler-image-in-picture',
                'svg'              => null,
                'order'            => 3,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.creative-suite.index',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('creative-suite'),
                'badge'            => 'new',
            ],
            'ext_chat_bot' => [
                'parent_key'       => null,
                'key'              => 'ext_chat_bot',
                'route'            => 'dashboard.chatbot.index',
                'label'            => 'AI Chat Bots',
                'data-name'        => Introduction::AI_EXT_CHATBOT,
                'icon'             => 'tabler-message-chatbot',
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.chatbot.*',
                ],
                'show_condition' => Route::has('dashboard.chatbot.index'),
                'badge'          => 'new',
            ],
            'ext_chatbot_knowledge_base_article' => [
                'parent_key'       => null,
                'key'              => 'ext_chatbot_knowledge_base_article',
                'route'            => 'dashboard.chatbot.knowledge-base-article.index',
                'label'            => 'AI Bot Knowledge Base',
                'data-name'        => Introduction::AI_EXT_CHATBOT_KNOWLEDGE_BASE_ARTICLES,
                'icon'             => 'tabler-library',
                'svg'              => null,
                'order'            => 5,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.chatbot.*',
                ],
                'show_condition' => Route::has('dashboard.chatbot.knowledge-base-article.index'),
            ],
            'ext_chatbot_chatbot_customer_article' => [
                'parent_key'       => null,
                'key'              => 'ext_chatbot_chatbot_customer_article',
                'route'            => 'dashboard.chatbot.chatbot-customer.index',
                'label'            => 'AI Bot Contacts',
                'data-name'        => Introduction::AI_EXT_CHATBOT_CUSTOMER,
                'icon'             => 'tabler-library',
                'svg'              => null,
                'order'            => 5,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.chatbot.*',
                ],
                'show_condition' => Route::has('dashboard.chatbot.chatbot-customer.index'),
            ],
            'ext_voice_chatbot' => [
                'parent_key'       => null,
                'key'              => 'ext_voice_chatbot',
                'route'            => 'dashboard.chatbot-voice.index',
                'label'            => 'AI Voice Bots',
                'data-name'        => Introduction::AI_EXT_VOICE_CHATBOT,
                'icon'             => 'tabler-message-chatbot',
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.chatbot-voice.*',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('chatbot-voice'),
                'badge'            => 'new',
            ],
            'marketing_bot_dashboard' => [
                'parent_key'       => null,
                'key'              => 'marketing_bot_dashboard',
                'route'            => 'dashboard.user.marketing-bot.dashboard',
                'label'            => 'Marketing Bot',
                'icon'             => 'tabler-dashboard',
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.user.marketing-bot.dashboard'],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
                'is_admin'         => false,
                'badge'            => 'new',
            ],
            'marketing_bot_settings' => [
                'parent_key'       => null,
                'key'              => 'marketing_bot_settings',
                'route'            => 'dashboard.user.marketing-bot.settings.index',
                'label'            => 'Marketing Bot Settings',
                'icon'             => 'tabler-settings-code',
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.user.marketing-bot.settings.index'],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
                'is_admin'         => false,
            ],
            'marketing_bot' => [
                'parent_key'       => null,
                'key'              => 'marketing_bot',
                'route'            => null,
                'label'            => 'Marketing bot',
                'icon'             => null,
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'label',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
            ],
            'marketing_bot_inbox' => [
                'parent_key'       => null,
                'key'              => 'marketing_bot_inbox',
                'route'            => 'dashboard.user.marketing-bot.inbox.index',
                'label'            => 'Inbox',
                'icon'             => 'tabler-inbox',
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.marketing-bot.inbox.*',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
            ],
            'marketing_bot_campaigns' => [
                'parent_key'       => null,
                'key'              => 'marketing_bot_campaigns',
                'route'            => 'dashboard.user.marketing-bot.whatsapp-campaign.index',
                'label'            => 'Campaigns',
                'icon'             => 'tabler-flag-share',
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.marketing-bot.whatsapp-campaign.*', 'dashboard.user.marketing-bot.telegram-campaign.*',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
            ],
            'marketing_bot_campaigns_whatsapp' => [
                'parent_key'       => 'marketing_bot_campaigns',
                'key'              => 'marketing_bot_campaigns_whatsapp',
                'route'            => 'dashboard.user.marketing-bot.whatsapp-campaign.index',
                'label'            => 'Whatsapp',
                'icon'             => null,
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.user.marketing-bot.whatsapp-group.*'],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
            ],

            'marketing_bot_campaigns_telegram' => [
                'parent_key'       => 'marketing_bot_campaigns',
                'key'              => 'marketing_bot_campaigns_telegram',
                'route'            => 'dashboard.user.marketing-bot.telegram-campaign.index',
                'label'            => 'Telegram',
                'icon'             => null,
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.user.marketing-bot.telegram-campaign.*'],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
            ],

            'marketing_bot_telegram' => [
                'parent_key'       => null,
                'key'              => 'marketing_bot_telegram',
                'route'            => 'dashboard.user.marketing-bot.telegram-group.index',
                'label'            => 'Telegram',
                'icon'             => 'tabler-brand-telegram',
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.marketing-bot.telegram-group.*',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
            ],

            'marketing_bot_telegram_group' => [
                'parent_key'       => 'marketing_bot_telegram',
                'key'              => 'marketing_bot_telegram_group',
                'route'            => 'dashboard.user.marketing-bot.telegram-group.index',
                'label'            => 'Telegram Groups',
                'icon'             => null,
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.user.marketing-bot.telegram-group.*'],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
            ],

            'marketing_bot_telegram_subscribers' => [
                'parent_key'       => 'marketing_bot_telegram',
                'key'              => 'marketing_bot_telegram_subscribers',
                'route'            => 'dashboard.user.marketing-bot.telegram-subscriber.index',
                'label'            => 'Telegram Subscribers',
                'icon'             => null,
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.user.marketing-bot.telegram-subscriber.*'],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
            ],

            'marketing_bot_whatsapp' => [
                'parent_key'       => null,
                'key'              => 'marketing_bot_whatsapp',
                'route'            => 'dashboard.user.marketing-bot.contact.index',
                'label'            => 'Whatsapp',
                'icon'             => 'tabler-brand-whatsapp',
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.user.marketing-bot.contact.*'],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
            ],
            'marketing_bot_whatsapp_contact' => [
                'parent_key'       => 'marketing_bot_whatsapp',
                'key'              => 'marketing_bot_whatsapp_contact',
                'route'            => 'dashboard.user.marketing-bot.contact.index',
                'label'            => 'Contact Lists',
                'icon'             => null,
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.marketing-bot.contact.index',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
            ],
            'marketing_bot_whatsapp_segment' => [
                'parent_key'       => 'marketing_bot_whatsapp',
                'key'              => 'marketing_bot_whatsapp_segment',
                'route'            => 'dashboard.user.marketing-bot.segment.index',
                'label'            => 'Segments',
                'icon'             => null,
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.marketing-bot.contact.index',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
            ],
            'marketing_bot_whatsapp_contact_list' => [
                'parent_key'       => 'marketing_bot_whatsapp',
                'key'              => 'marketing_bot_whatsapp_contact_list',
                'route'            => 'dashboard.user.marketing-bot.contact-list.index',
                'label'            => 'Contacts',
                'icon'             => null,
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.marketing-bot.contact-list.index',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('marketing-bot'),
            ],
            'ext_chat_bot_agent' => [
                'parent_key'       => null,
                'key'              => 'ext_chat_bot_agent',
                'route'            => 'dashboard.chatbot-agent.index',
                'label'            => 'Human Agent',
                'data-name'        => Introduction::AI_EXT_CHATBOT_AGENT,
                'icon'             => 'tabler-message',
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.chatbot-agent.*',
                ],
                'show_condition'   => Route::has('dashboard.chatbot-agent.index'),
                'badge'            => 'new',
            ],

            'ext_social_media_dropdown' => [
                'parent_key'       => null,
                'key'              => 'ext_social_media_dropdown',
                'route'            => 'dashboard.user.social-media.index',
                'label'            => 'AI Social Media',
                'data-name'        => Introduction::SOCIAL_MEDIA,
                'icon'             => 'tabler-thumb-up',
                'svg'              => null,
                'order'            => 5,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.social-media.*',
                ],
                'show_condition'   => Route::has('dashboard.user.social-media.index') && MarketplaceHelper::isRegistered('social-media'),
                'badge'            => 'new',
            ],
            'ext_social_media' => [
                'parent_key'       => 'ext_social_media_dropdown',
                'key'              => 'ext_social_media',
                'route'            => 'dashboard.user.social-media.index',
                'label'            => 'Dashboard',
                'data-name'        => null,
                'icon'             => null,
                'svg'              => null,
                'order'            => 1,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.social-media.index',
                ],
                'show_condition' => Route::has('dashboard.user.social-media.index'),
            ],
            'ext_social_media_campaign' => [
                'parent_key'       => 'ext_social_media_dropdown',
                'key'              => 'ext_social_media_campaign',
                'route'            => 'dashboard.user.social-media.campaign.index',
                'label'            => 'Campaigns',
                'data-name'        => null,
                'icon'             => null,
                'svg'              => null,
                'order'            => 3,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.social-media.campaign.*',
                ],
                'show_condition' => Route::has('dashboard.user.social-media.index'),
            ],
            'ext_social_media_platform' => [
                'parent_key'       => 'ext_social_media_dropdown',
                'key'              => 'ext_social_media_platform',
                'route'            => 'dashboard.user.social-media.platforms',
                'label'            => 'Platforms',
                'data-name'        => null,
                'icon'             => null,
                'svg'              => null,
                'order'            => 3,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.social-media.platforms',
                    'dashboard.user.social-media.platforms.*',
                ],
                'show_condition' => Route::has('dashboard.user.social-media.platforms'),
            ],
            'ext_social_media_post' => [
                'parent_key'       => 'ext_social_media_dropdown',
                'key'              => 'ext_social_media_post',
                'route'            => 'dashboard.user.social-media.post.index',
                'label'            => 'Social Media Posts',
                'data-name'        => null,
                'icon'             => null,
                'svg'              => null,
                'order'            => 3,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.social-media.post.*',
                ],
                'show_condition' => Route::has('dashboard.user.social-media.post.index'),
            ],
            'ext_social_media_calendar' => [
                'parent_key'       => 'ext_social_media_dropdown',
                'key'              => 'ext_social_media_calendar',
                'route'            => 'dashboard.user.social-media.calendar',
                'label'            => 'Calendar',
                'data-name'        => null,
                'icon'             => null,
                'svg'              => null,
                'order'            => 3,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.social-media.calendar',
                ],
                'show_condition' => Route::has('dashboard.user.social-media.calendar'),
            ],
            'ai_influencer' => [
                'parent_key'       => null,
                'key'              => 'ai_influencer',
                'route'            => 'dashboard.user.ai-influencer.index',
                'label'            => 'AI Influencer',
                'data-name'        => Introduction::AI_INFLUENCER,
                'icon'             => 'tabler-star',
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.ai-influencer.index',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('url-to-video') || MarketplaceHelper::isRegistered('influencer-avatar') || MarketplaceHelper::isRegistered('ai-viral-clips'),
                'badge'            => 'new',
            ],
            'url_to_video' => [
                'parent_key'       => null,
                'key'              => 'url_to_video',
                'route'            => 'dashboard.user.url-to-video.index',
                'label'            => 'Url To Video',
                'data-name'        => Introduction::URL_TO_VIDEO,
                'icon'             => 'tabler-photo-video',
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.url-to-video.index',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('url-to-video'),
                'badge'            => 'new',
            ],
            'viral_clips' => [
                'parent_key'       => null,
                'key'              => 'viral_clips',
                'route'            => 'dashboard.user.viral-clips.index',
                'label'            => 'AI Viral Clips',
                'data-name'        => Introduction::VIRAL_CLIPS,
                'icon'             => 'tabler-movie',
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.viral-clips.index',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('ai-viral-clips'),
                'badge'            => 'new',
            ],
            'influencer_avatar' => [
                'parent_key'       => null,
                'key'              => 'influencer_avatar',
                'route'            => 'dashboard.user.influencer-avatar.index',
                'label'            => 'Influencer Avatar',
                'data-name'        => Introduction::INFLUENCER_AVATAR,
                'icon'             => 'tabler-device-mobile-star',
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.influencer-avatar.index',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('influencer-avatar'),
                'badge'            => 'new',
            ],
            'documents' => [
                'parent_key'       => null,
                'key'              => 'documents',
                'route'            => 'dashboard.user.openai.documents.all',
                'label'            => 'Documents',
                'data-name'        => Introduction::AI_DOCUMENT,
                'icon'             => 'tabler-archive',
                'svg'              => null,
                'order'            => 3,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.openai.documents.*',
                ],
                'show_condition' => true,
            ],
            'ai_editor' => [
                'parent_key'       => null,
                'key'              => 'ai_editor',
                'route'            => 'dashboard.user.generator.index',
                'label'            => 'AI Editor',
                'data-name'        => Introduction::AI_EDITOR,
                'icon'             => 'tabler-notebook',
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.generator.index',
                ],
                'show_condition' => (bool) Helper::setting('feature_ai_advanced_editor', null, $setting),
            ],
            'ai_writer' => [
                'parent_key'       => null,
                'key'              => 'ai_writer',
                'route'            => 'dashboard.user.openai.list',
                'label'            => 'AI Writer',
                'data-name'        => Introduction::AI_WRITER,
                'icon'             => 'tabler-notes',
                'svg'              => null,
                'order'            => 5,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.openai.list', 'dashboard.user.openai.generator.*', 'dashboard.user.openai.generator.workbook',
                ],
                'show_condition' => (bool) Helper::setting('feature_ai_writer', null, $setting),
            ],
            'ai_video' => [
                'parent_key'       => null,
                'key'              => 'ai_video',
                'route'            => 'dashboard.user.openai.generator',
                'route_slug'       => 'ai_video',
                'label'            => 'AI Video',
                'data-name'        => Introduction::AI_VIDEO,
                'icon'             => 'tabler-video',
                'svg'              => null,
                'order'            => 6,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::settingTwo('feature_ai_video', null, $settings_two),
            ],
            'ai_video_to_video' => [
                'parent_key'       => null,
                'key'              => 'ai_video_to_video',
                'route'            => 'dashboard.user.openai.generator',
                'route_slug'       => 'ai_video_to_video',
                'label'            => 'AI Video To Video',
                'data-name'        => Introduction::AI_VIDEO_TO_VIDEO,
                'icon'             => 'tabler-video',
                'svg'              => null,
                'order'            => 7,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => MarketplaceHelper::isRegistered('ai-video-to-video'),
            ],
            'ai_image_generator' => [
                'parent_key'       => null,
                'key'              => 'ai_image_generator',
                'route'            => 'dashboard.user.openai.generator',
                'route_slug'       => 'ai_image_generator',
                'label'            => 'AI Image',
                'data-name'        => Introduction::AI_IMAGE,
                'icon'             => 'tabler-photo',
                'svg'              => null,
                'order'            => 7,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_image', null, $setting),
            ],
            'ai_article_wizard' => [
                'parent_key'       => null,
                'key'              => 'ai_article_wizard',
                'route'            => 'dashboard.user.openai.articlewizard.new',
                'route_slug'       => 'ai_article_wizard',
                'label'            => 'AI Article Wizard',
                'data-name'        => Introduction::AI_ARTICLE_WIZARD,
                'icon'             => 'tabler-ad-2',
                'svg'              => null,
                'order'            => 8,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.openai.articlewizard.new',
                ],
                'show_condition' => (bool) Helper::setting('feature_ai_article_wizard', null, $setting),
            ],
            'ai_pdf' => [
                'parent_key'       => null,
                'key'              => 'ai_pdf',
                'route'            => 'dashboard.user.openai.generator.workbook',
                'route_slug'       => 'ai_pdf',
                'label'            => 'AI File Chat',
                'data-name'        => Introduction::AI_PDF,
                'icon'             => 'tabler-file-pencil',
                'svg'              => null,
                'order'            => 9,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_pdf', null, $setting),
            ],
            'ai_vision' => [
                'parent_key'       => null,
                'key'              => 'ai_vision',
                'route'            => 'dashboard.user.openai.generator.workbook',
                'route_slug'       => 'ai_vision',
                'label'            => 'AI Vision',
                'data-name'        => Introduction::AI_VISION,
                'icon'             => 'tabler-scan-eye',
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_vision', null, $setting),
            ],
            'ai_realtime_voice_chat' => [
                'parent_key'       => null,
                'key'              => 'ai_realtime_voice_chat',
                'route'            => 'dashboard.user.openai.chat.chat',
                'route_slug'       => 'ai_realtime_voice_chat',
                'label'            => 'AI Realtime Voice Chat',
                'data-name'        => Introduction::AI_REALTIME_VOICE_CHAT,
                'icon'             => 'tabler-wave-sine',
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => setting('realtime_voice_chat', 0) == 1 && MarketplaceHelper::isRegistered('openai-realtime-chat'),
            ],
            'ai_realtime_image' => [
                'parent_key'       => null,
                'key'              => 'ai_realtime_image',
                'route'            => 'dashboard.user.ai-realtime-image.index',
                'route_slug'       => 'ai_realtime_image',
                'label'            => 'AI Realtime Image',
                'data-name'        => Introduction::AI_REALTIME_IMAGE,
                'icon'             => 'tabler-image-in-picture',
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => setting('ai_realtime_image', '0') == '1' && MarketplaceHelper::isRegistered('ai-realtime-image'),
            ],
            'ai_rewriter' => [
                'parent_key'       => null,
                'key'              => 'ai_rewriter',
                'route'            => 'dashboard.user.openai.rewriter',
                'route_slug'       => 'ai_rewriter',
                'label'            => 'AI ReWriter',
                'data-name'        => Introduction::AI_REWRITER,
                'icon'             => 'tabler-ballpen',
                'svg'              => null,
                'order'            => 11,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.openai.rewriter', 'dashboard.user.openai.rewriter.*',
                ],
                'show_condition' => (bool) Helper::setting('feature_ai_rewriter', null, $setting),
            ],
            'ai_chat_image' => [
                'parent_key'       => null,
                'key'              => 'ai_chat_image',
                'route'            => 'dashboard.user.openai.generator.workbook',
                'route_slug'       => 'ai_chat_image',
                'label'            => 'AI Chat Image',
                'data-name'        => Introduction::AI_CHAT_IMAGE,
                'icon'             => 'tabler-photo',
                'svg'              => null,
                'order'            => 12,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => route('dashboard.user.openai.generator.workbook', 'ai_chat_image') === url()->current(),
                'show_condition'   => (bool) Helper::setting('feature_ai_chat_image', null, $setting),
            ],
            'ai_chat_all' => [
                'parent_key'       => null,
                'key'              => 'ai_chat_all',
                'route'            => Helper::appIsDemo() ? 'dashboard.user.openai.chat.list' : 'dashboard.user.openai.chat.chat',
                'route_slug'       => null,
                'label'            => 'AI Chat',
                'data-name'        => Introduction::AI_CHAT,
                'icon'             => 'tabler-message-dots',
                'svg'              => null,
                'order'            => 13,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.openai.chat.*',
                ],
                'show_condition'   => (bool) Helper::setting('feature_ai_chat', null, $setting),
            ],
            'ai_chat_pro' => [
                'parent_key'       => null,
                'key'              => 'ai_chat_pro',
                'route'            => 'dashboard.user.openai.chat.pro.index',
                'route_slug'       => null,
                'label'            => 'AI Chat Pro',
                'data-name'        => Introduction::AI_CHAT_PRO,
                'icon'             => 'tabler-message-plus',
                'svg'              => null,
                'order'            => 13,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.openai.chat.pro.*',
                ],
                'show_condition' => MarketplaceHelper::isRegistered('ai-chat-pro') && in_array(setting('ai_chat_display_type', 'menu'), ['menu', 'both_fm']),
            ],
            'ai_code_generator' => [
                'parent_key'       => null,
                'key'              => 'ai_code_generator',
                'route'            => 'dashboard.user.openai.generator.workbook',
                'route_slug'       => 'ai_code_generator',
                'label'            => 'AI Code',
                'data-name'        => Introduction::AI_CODE,
                'icon'             => 'tabler-terminal-2',
                'svg'              => null,
                'order'            => 14,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_code', null, $setting),
            ],
            'ai_youtube' => [
                'parent_key'       => null,
                'key'              => 'ai_youtube',
                'route'            => 'dashboard.user.openai.generator.workbook',
                'route_slug'       => 'ai_youtube',
                'label'            => 'AI YouTube',
                'data-name'        => Introduction::AI_YOUTUBE,
                'icon'             => 'tabler-brand-youtube',
                'svg'              => null,
                'order'            => 15,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_youtube', null, $setting),
            ],
            'ai_rss' => [
                'parent_key'       => null,
                'key'              => 'ai_rss',
                'route'            => 'dashboard.user.openai.generator.workbook',
                'route_slug'       => 'ai_rss',
                'label'            => 'AI RSS',
                'data-name'        => Introduction::AI_RSS,
                'icon'             => 'tabler-rss',
                'svg'              => null,
                'order'            => 15,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_rss', null, $setting),
            ],
            'ai_speech_to_text' => [
                'parent_key'       => null,
                'key'              => 'ai_speech_to_text',
                'route'            => 'dashboard.user.openai.generator',
                'route_slug'       => 'ai_speech_to_text',
                'label'            => 'AI Speech to Text',
                'data-name'        => Introduction::AI_SPEECH_TO_TEXT,
                'icon'             => 'tabler-microphone',
                'svg'              => null,
                'order'            => 16,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_speech_to_text', null, $setting),
            ],
            'ai_voiceover' => [
                'parent_key'       => null,
                'key'              => 'ai_voiceover',
                'route'            => 'dashboard.user.openai.generator',
                'route_slug'       => 'ai_voiceover',
                'label'            => 'AI Voiceover',
                'data-name'        => Introduction::AI_VOICEOVER,
                'icon'             => 'tabler-volume',
                'svg'              => null,
                'order'            => 17,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_voiceover', null, $setting),
            ],
            'ai_voice_isolator' => [
                'parent_key'       => null,
                'key'              => 'ai_voice_isolator',
                'route'            => 'dashboard.user.openai.generator',
                'route_slug'       => 'ai_voice_isolator',
                'label'            => 'AI Voice Isolator',
                'data-name'        => Introduction::AI_VOICE_ISOLATOR,
                'icon'             => 'tabler-ear-scan',
                'svg'              => null,
                'order'            => 18,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => null,
                'show_condition'   => setting('ai_voice_isolator', '1') == '1' && MarketplaceHelper::isRegistered('voice-isolator'),
            ],
            'ai_voiceover_clone' => [
                'parent_key'       => null,
                'key'              => 'ai_voiceover_clone',
                'route'            => 'dashboard.user.voice.index',
                'route_slug'       => null,
                'label'            => 'AI Voice Clone',
                'data-name'        => Introduction::AI_VOICEOVER_CLONE,
                'icon'             => 'tabler-microphone-2',
                'svg'              => null,
                'order'            => 18,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.voice.*',
                ],
                'show_condition' => Helper::setting('feature_ai_voice_clone', null, $setting) && Helper::settingTwo('elevenlabs_api_key', null, $settings_two),
            ],
            'team_menu' => [
                'parent_key'       => null,
                'key'              => 'team_menu',
                'route'            => 'dashboard.user.team.index',
                'route_slug'       => null,
                'label'            => 'Team',
                'data-name'        => Introduction::TEAM_MENU,
                'icon'             => 'tabler-user-plus',
                'svg'              => null,
                'order'            => 19,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.team.*',
                ],
                'show_condition' => $this->showTeamFunctionality(),
            ],
            'brand_voice' => [
                'parent_key'       => null,
                'key'              => 'brand_voice',
                'route'            => 'dashboard.user.brand.index',
                'route_slug'       => null,
                'label'            => 'Brand Voice',
                'data-name'        => Introduction::BRAND_VOICE,
                'icon'             => 'tabler-brand-trello',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.brand.*',
                ],
                'show_condition'   => true,
            ],
            'advanced_image' => [
                'parent_key'       => null,
                'key'              => 'advanced_image',
                'route'            => 'dashboard.user.advanced-image.index',
                'route_slug'       => null,
                'label'            => 'AI Image Editor',
                'data-name'        => Introduction::ADVANCED_IMAGE,
                'icon'             => 'tabler-photo-edit',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => null,
                'show_condition'   => Route::has('dashboard.user.advanced-image.index'),
                'badge'            => 'new',
            ],
            'ai_avatar' => [
                'parent_key'       => null,
                'key'              => 'ai_avatar',
                'route'            => 'dashboard.user.ai-avatar.index',
                'route_slug'       => null,
                'label'            => 'AI Avatar',
                'data-name'        => Introduction::AI_AVATAR,
                'icon'             => 'tabler-slideshow',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => null,
                'show_condition'   => Route::has('dashboard.user.ai-avatar.index'),
            ],
            'ai_persona' => [
                'parent_key'       => null,
                'key'              => 'ai_persona',
                'route'            => 'dashboard.user.ai-persona.index',
                'route_slug'       => null,
                'label'            => 'AI Persona',
                'data-name'        => Introduction::AI_AVATAR_PRO,
                'icon'             => 'tabler-camera-star',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => null,
                'show_condition'   => Route::has('dashboard.user.ai-persona.index'),
            ],
            'ai_video_pro' => [
                'parent_key'       => null,
                'key'              => 'ai_video_pro',
                'route'            => 'dashboard.user.ai-video-pro.index',
                'route_slug'       => null,
                'label'            => 'AI Video Pro',
                'data-name'        => Introduction::AI_FALL_VIDEO,
                'icon'             => 'tabler-video',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => false,
                'show_condition'   => Route::has('dashboard.user.ai-video-pro.index'),
            ],
            'ai_music' => [
                'parent_key'       => null,
                'key'              => 'ai_music',
                'route'            => 'dashboard.user.ai-music.index',
                'route_slug'       => null,
                'label'            => 'AI Music',
                'data-name'        => Introduction::AI_MUSIC,
                'icon'             => 'tabler-slideshow',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => null,
                'show_condition'   => Route::has('dashboard.user.ai-music.index'),
            ],
            'ext_ai_music_pro' => [
                'parent_key'       => null,
                'key'              => 'ext_ai_music_pro',
                'route'            => 'dashboard.user.ai-music-pro.index',
                'route_slug'       => null,
                'label'            => 'AI Music Pro',
                'data-name'        => Introduction::AI_MUSIC_PRO,
                'icon'             => 'tabler-music',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => null,
                'show_condition'   => MarketplaceHelper::isRegistered('ai-music-pro'),
                'badge'            => 'new',
            ],
            'ai_product_shot' => [
                'parent_key'       => null,
                'key'              => 'ai_product_shot',
                'route'            => 'dashboard.user.ai-product-shot.index',
                'route_slug'       => null,
                'label'            => 'AI Product Photography',
                'data-name'        => Introduction::AI_PRODUCT_SHOT,
                'icon'             => 'tabler-photo-star',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => null,
                'show_condition'   => Route::has('dashboard.user.ai-product-shot.index'),
            ],
            'api_keys' => [
                'parent_key'       => null,
                'key'              => 'api_keys',
                'route'            => 'dashboard.user.apikeys.index',
                'route_slug'       => null,
                'label'            => 'API Keys',
                'data-name'        => Introduction::USER_API_KEYS,
                'icon'             => 'tabler-key',
                'svg'              => null,
                'order'            => 21,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.apikeys.*',
                ],
                'show_condition' => Helper::appIsDemo() || Helper::setting('user_api_option', null, $setting) || auth()->user()?->relationPlan?->getAttribute('user_api'),
            ],
            'affiliates' => [
                'parent_key'       => null,
                'key'              => 'affiliates',
                'route'            => 'dashboard.user.affiliates.index',
                'route_slug'       => null,
                'label'            => 'Affiliates',
                'data-name'        => Introduction::AFFILIATES,
                'icon'             => 'tabler-currency-dollar',
                'svg'              => null,
                'order'            => 22,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => Helper::setting('feature_affilates', null, $setting) && (\auth()->user()?->affiliate_status === 1),
            ],
            'support' => [
                'parent_key'       => null,
                'key'              => 'support',
                'route'            => 'dashboard.support.list',
                'route_slug'       => null,
                'label'            => 'Support',
                'data-name'        => Introduction::SUPPORT,
                'icon'             => 'tabler-lifebuoy',
                'svg'              => null,
                'order'            => 23,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => true,
            ],
            'integration' => [
                'parent_key'       => null,
                'key'              => 'integration',
                'route'            => 'dashboard.user.integration.index',
                'route_slug'       => null,
                'label'            => 'Integration',
                'data-name'        => Introduction::INTEGRATION,
                'icon'             => 'tabler-webhook',
                'svg'              => null,
                'order'            => 24,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.integration.*',
                ],
                'show_condition' => Integration::query()->whereHas('hasExtension')->count(),
            ],
            'divider_one' => [
                'parent_key'       => null,
                'key'              => 'divider_one',
                'route'            => null,
                'label'            => null,
                'icon'             => null,
                'svg'              => null,
                'order'            => 25,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'divider',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
            ],
            'links' => [
                'parent_key'       => null,
                'key'              => 'links',
                'route'            => null,
                'label'            => 'Links',
                'icon'             => null,
                'svg'              => null,
                'order'            => 26,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'label',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
            ],
            'favorites' => [
                'parent_key'       => null,
                'key'              => 'favorites',
                'route'            => 'dashboard.user.openai.list',
                'route_slug'       => 'filter=favorite',
                'label'            => 'Favorites',
                'icon'             => null,
                'svg'              => null,
                'order'            => 27,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
                'letter_icon'      => 1,
                'letter_icon_bg'   => 'bg-[#7A8193] text-white',
            ],
            'workbook' => [
                'parent_key'       => null,
                'key'              => 'workbook',
                'route'            => 'dashboard.user.openai.documents.all',
                'route_slug'       => '?filter=favorites',
                'label'            => 'Workbook',
                'icon'             => null,
                'svg'              => null,
                'order'            => 28,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
                'letter_icon'      => 1,
                'letter_icon_bg'   => 'bg-[#658C8E] text-white',
            ],
            'divider_two' => [
                'parent_key'       => null,
                'key'              => 'divider_two',
                'route'            => null,
                'label'            => null,
                'icon'             => null,
                'svg'              => null,
                'order'            => 29,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'divider',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'admin_label' => [
                'parent_key'       => null,
                'key'              => 'admin_label',
                'route'            => null,
                'label'            => 'Admin',
                'icon'             => null,
                'svg'              => null,
                'order'            => 30,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'label',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'admin_dashboard' => [
                'parent_key'       => null,
                'key'              => 'admin_dashboard',
                'route'            => 'dashboard.admin.index',
                'label'            => 'Dashboard',
                'data-name'        => Introduction::ADMIN_DASHBOARD,
                'icon'             => 'tabler-layout-2',
                'svg'              => null,
                'order'            => 31,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.index',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'marketplace' => [
                'parent_key'       => null,
                'key'              => 'marketplace',
                'route'            => 'dashboard.admin.marketplace.index',
                'label'            => 'Marketplace',
                'data-name'        => Introduction::ADMIN_MARKETPLACE,
                'icon'             => 'tabler-building-store',
                'svg'              => null,
                'order'            => 32,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.marketplace.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'themes' => [
                'parent_key'       => null,
                'key'              => 'themes',
                'route'            => 'dashboard.admin.themes.index',
                'label'            => 'Themes',
                'data-name'        => Introduction::ADMIN_THEMES,
                'icon'             => 'tabler-palette',
                'svg'              => null,
                'order'            => 33,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.themes.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'ext_migration' => [
                'parent_key'       => null,
                'key'              => 'ext_migration',
                'route'            => 'migration::welcome',
                'label'            => 'Migration',
                'data-name'        => Introduction::MIGRATION,
                'icon'             => 'tabler-transfer-in',
                'svg'              => null,
                'order'            => 33,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'migration::*',
                ],
                'show_condition' => MarketplaceHelper::isRegistered('migration'),
                'is_admin'       => true,
                'badge'          => 'new',
            ],
            'user_management' => [
                'parent_key'       => null,
                'key'              => 'user_management',
                'route'            => 'dashboard.admin.users.index',
                'label'            => 'User Management',
                'data-name'        => Introduction::ADMIN_USER_MANAGEMENT,
                'icon'             => 'tabler-users',
                'svg'              => null,
                'order'            => 34,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.users.*',
                ],
                'show_condition'     => true,
                'is_admin'           => true,
                'admin_permission'   => 'user_management',
            ],
            'user_list' => [
                'parent_key'       => 'user_management',
                'key'              => 'user_list',
                'route'            => 'dashboard.admin.users.index',
                'label'            => 'Users List',
                'icon'             => null,
                'svg'              => null,
                'order'            => 34,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.users.index',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'user_activity' => [
                'parent_key'       => 'user_management',
                'key'              => 'user_activity',
                'route'            => 'dashboard.admin.users.activity',
                'label'            => 'Users Activities',
                'icon'             => null,
                'svg'              => null,
                'order'            => 34,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.users.activity',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'user_dashboard' => [
                'parent_key'       => 'user_management',
                'key'              => 'user_dashboard',
                'route'            => 'dashboard.admin.users.dashboard',
                'label'            => 'Users Dashboard',
                'icon'             => null,
                'svg'              => null,
                'order'            => 34,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.users.dashboard',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'user_deletion' => [
                'parent_key'       => 'user_management',
                'key'              => 'user_deletion',
                'route'            => 'dashboard.admin.users.deletion.reqs',
                'label'            => 'User Deletion Requests',
                'icon'             => null,
                'svg'              => null,
                'order'            => 34,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.users.deletion.reqs',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'user_permission' => [
                'parent_key'       => 'user_management',
                'key'              => 'user_permission',
                'route'            => 'dashboard.admin.users.permissions',
                'label'            => 'User Permissions',
                'icon'             => null,
                'svg'              => null,
                'order'            => 34,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.users.permissions',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'announcements' => [
                'parent_key'       => null,
                'key'              => 'announcements',
                'route'            => 'dashboard.admin.announcements.index',
                'label'            => 'Announcements',
                'data-name'        => Introduction::ADMIN_ANNOUNCEMENTS,
                'icon'             => 'tabler-speakerphone',
                'svg'              => null,
                'order'            => 35,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.announcements.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'discount-manager' => [
                'parent_key'       => null,
                'key'              => 'discount-manager',
                'route'            => 'dashboard.admin.discount-manager.index',
                'label'            => 'Discount & Offer Manager',
                'data-name'        => Introduction::DISCOUNT_MANAGER,
                'icon'             => 'tabler-shopping-bag-discount',
                'svg'              => null,
                'order'            => 37,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.discount-manager.*',
                ],
                'show_condition' => MarketplaceHelper::isRegistered('discount-manager'),
                'is_admin'       => true,
                'badge'          => 'new',
            ],
            'site_promo' => [
                'parent_key'       => null,
                'key'              => 'site_promo',
                'route'            => 'dashboard.admin.ads.index',
                'label'            => 'Google adsense',
                'data-name'        => Introduction::ADMIN_GOOGLE_ADSENSE,
                'icon'             => 'tabler-ad-circle',
                'svg'              => null,
                'order'            => 35,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.ads.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'support_requests' => [
                'parent_key'       => null,
                'key'              => 'support_requests',
                'route'            => 'dashboard.support.list',
                'label'            => 'Support Requests',
                'data-name'        => Introduction::ADMIN_SUPPORT_REQUEST,
                'icon'             => 'tabler-lifebuoy',
                'svg'              => null,
                'order'            => 36,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.support.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'templates' => [
                'parent_key'       => null,
                'key'              => 'templates',
                'route'            => 'dashboard.admin.openai.list',
                'label'            => 'Templates',
                'data-name'        => Introduction::ADMIN_TEMPLATES,
                'icon'             => 'tabler-list-details',
                'svg'              => null,
                'order'            => 37,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'built_in_templates' => [
                'parent_key'       => 'templates',
                'key'              => 'built_in_templates',
                'route'            => 'dashboard.admin.openai.list',
                'label'            => 'Built-in Templates',
                'icon'             => null,
                'svg'              => null,
                'order'            => 38,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.openai.list',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'custom_templates' => [
                'parent_key'       => 'templates',
                'key'              => 'custom_templates',
                'route'            => 'dashboard.admin.openai.custom.list',
                'label'            => 'Custom Templates',
                'icon'             => null,
                'svg'              => null,
                'order'            => 39,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.openai.custom.list',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'ai_writer_categories' => [
                'parent_key'       => 'templates',
                'key'              => 'ai_writer_categories',
                'route'            => 'dashboard.admin.openai.categories.list',
                'label'            => 'AI Writer Categories',
                'icon'             => null,
                'svg'              => null,
                'order'            => 40,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.openai.categories.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'chat_settings' => [
                'parent_key'       => null,
                'key'              => 'chat_settings',
                'route'            => 'dashboard.admin.openai.chat.category',
                'label'            => 'Chat Settings',
                'data-name'		      => Introduction::ADMIN_CHAT_SETTINGS,
                'icon'             => 'tabler-message-circle',
                'svg'              => null,
                'order'            => 41,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.chatbot.*', 'dashboard.admin.openai.chat.list',
                    'dashboard.admin.openai.chat.addOrUpdate', 'dashboard.admin.openai.chat.category',
                    'dashboard.admin.openai.chat.addOrUpdateCategory', 'dashboard.admin.ai-chat-model.index',
                    'dashboard.admin.ai-assistant.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'chat_categories' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'chat_categories',
                'route'            => 'dashboard.admin.openai.chat.category',
                'label'            => 'Chat Categories',
                'icon'             => null,
                'svg'              => null,
                'order'            => 42,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.openai.chat.category.*', 'dashboard.admin.openai.chat.category',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'chat_templates' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'chat_templates',
                'route'            => 'dashboard.admin.openai.chat.list',
                'label'            => 'Chat Templates',
                'icon'             => null,
                'svg'              => null,
                'order'            => 43,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.openai.chat.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'chatbot_training' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'chatbot_training',
                'route'            => 'dashboard.admin.chatbot.index',
                'label'            => 'Chatbot Training',
                'icon'             => null,
                'svg'              => null,
                'order'            => 44,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.chatbot.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'voice_chatbot_training' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'voice_chatbot_training',
                'route'            => 'dashboard.admin.voice-chatbot.index',
                'label'            => 'Voice Chatbot Training',
                'icon'             => null,
                'svg'              => null,
                'order'            => 44,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => false,
                'active_condition' => [
                    'dashboard.admin.voice-chatbot.*',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('elevenlabs-voice-chat'),
                'is_admin'         => true,
            ],
            'ai_assistant' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'ai_assistant',
                'route'            => 'dashboard.admin.ai-assistant.index',
                'label'            => 'Assistant Training',
                'icon'             => null,
                'svg'              => null,
                'order'            => 44,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.ai-assistant.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'ai_chat_models' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'ai_chat_models',
                'route'            => 'dashboard.admin.ai-chat-model.index',
                'label'            => 'AI Chat Models',
                'icon'             => null,
                'svg'              => null,
                'order'            => 44,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.ai-chat-model.index',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'floating_chat_settings' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'floating_chat_settings',
                'route'            => 'dashboard.admin.chatbot.setting',
                'label'            => 'Floating Chat Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 45,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.chatbot.setting',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'external_chat_settings' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'external_chat_settings',
                'route'            => 'dashboard.admin.chatbot.external_settings',
                'label'            => 'External Chat Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 46,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.chatbot.external_settings',
                ],
                'show_condition' => false,
                'is_admin'       => true,
            ],
            'frontend' => [
                'parent_key'       => null,
                'key'              => 'frontend',
                'route'            => 'dashboard.admin.frontend.settings',
                'label'            => 'Frontend',
                'data-name'        => Introduction::ADMIN_FRONTEND,
                'icon'             => 'tabler-device-laptop',
                'svg'              => null,
                'order'            => 47,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.testimonials.*', 'dashboard.admin.frontend.authsettings', 'dashboard.admin.frontend.settings', 'dashboard.admin.frontend.faq.*', 'dashboard.admin.frontend.curtain.*', 'dashboard.admin.frontend.content-box.*', 'dashboard.admin.frontend.tools.*', 'dashboard.admin.frontend.tools.*', 'dashboard.admin.channel-setting.*', 'dashboard.admin.frontend.future.*', 'dashboard.admin.frontend.whois.*', 'dashboard.admin.frontend.generatorlist.*', 'dashboard.admin.clients.*', 'dashboard.admin.howitWorks.*', 'dashboard.admin.whois.*', 'dashboard.admin.frontend.menusettings', 'dashboard.admin.frontend.sectionsettings',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'frontend_settings' => [
                'parent_key'       => 'frontend',
                'key'              => 'frontend_settings',
                'route'            => 'dashboard.admin.frontend.settings',
                'label'            => 'Frontend Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 48,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.settings',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'frontend_section_settings' => [
                'parent_key'       => 'frontend',
                'key'              => 'frontend_section_settings',
                'route'            => 'dashboard.admin.frontend.sectionsettings',
                'label'            => 'Frontend Section Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 49,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.sectionsettings',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'frontend_menu' => [
                'parent_key'       => 'frontend',
                'key'              => 'frontend_menu',
                'route'            => 'dashboard.admin.frontend.menusettings',
                'label'            => 'Menu',
                'icon'             => null,
                'svg'              => null,
                'order'            => 50,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.menusettings',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'social_media_accounts' => [
                'parent_key'       => 'frontend',
                'key'              => 'social_media_accounts',
                'route'            => 'dashboard.admin.frontend.socialmedia',
                'label'            => 'Social Media Accounts',
                'icon'             => null,
                'svg'              => null,
                'order'            => 50,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.socialmedia',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'auth_settings' => [
                'parent_key'       => 'frontend',
                'key'              => 'auth_settings',
                'route'            => 'dashboard.admin.frontend.authsettings',
                'label'            => 'Auth Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 52,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.authsettings',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'f_a_q' => [
                'parent_key'       => 'frontend',
                'key'              => 'f_a_q',
                'route'            => 'dashboard.admin.frontend.faq.index',
                'label'            => 'F.A.Q',
                'icon'             => null,
                'svg'              => null,
                'order'            => 53,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.faq.index',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'tools_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'tools_section',
                'route'            => 'dashboard.admin.frontend.tools.index',
                'label'            => 'Tools Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 54,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.tools.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'channels_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'channels_section',
                'route'            => 'dashboard.admin.frontend.channel-setting.index',
                'label'            => 'Channels Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 54,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.channel-setting.*',
                ],
                'show_condition' => setting('front_theme') === 'social-media-front',
                'is_admin'       => true,
            ],
            'content_box' => [
                'parent_key'       => 'frontend',
                'key'              => 'content_box',
                'route'            => 'dashboard.admin.frontend.content-box.index',
                'label'            => 'Content Box Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 54,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.content-box.*',
                ],
                'show_condition' => setting('front_theme') === 'social-media-front',
                'is_admin'       => true,
            ],
            'curtain_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'curtain_section',
                'route'            => 'dashboard.admin.frontend.curtain.index',
                'label'            => 'Curtain Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 54,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.curtain.*',
                ],
                'show_condition' => in_array(setting('front_theme'), ['social-media-front', 'marketing-bot']),
                'is_admin'       => true,
            ],
            'features_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'features_section',
                'route'            => 'dashboard.admin.frontend.future.index',
                'label'            => 'Features Section',
                'icon'             => 'tabler-list-details',
                'svg'              => null,
                'order'            => 55,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.future.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'testimonials_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'testimonials_section',
                'route'            => 'dashboard.admin.testimonials.index',
                'label'            => 'Testimonials Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 56,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.testimonials.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'clients_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'clients_section',
                'route'            => 'dashboard.admin.clients.index',
                'label'            => 'Clients Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 57,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.clients.index',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'how_it_works_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'how_it_works_section',
                'route'            => 'dashboard.admin.howitWorks.index',
                'label'            => 'How it Works Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 58,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.howitWorks.*'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'who_can_use_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'who_can_use_section',
                'route'            => 'dashboard.admin.frontend.whois.index',
                'label'            => 'Who Can Use Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 59,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.whois.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'generators_list_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'generators_list_section',
                'route'            => 'dashboard.admin.frontend.generatorlist.index',
                'label'            => 'Generators List Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 60,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.generatorlist.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'finance' => [
                'parent_key'       => null,
                'key'              => 'finance',
                'route'            => 'dashboard.admin.finance.plans.index',
                'label'            => 'Finance',
                'data-name'        => Introduction::ADMIN_FINANCE,
                'icon'             => 'tabler-wallet',
                'svg'              => null,
                'order'            => 61,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.*', 'dashboard.admin.bank.transactions.list',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'bank_transactions' => [
                'parent_key'       => 'finance',
                'key'              => 'bank_transactions',
                'route'            => 'dashboard.admin.bank.transactions.list',
                'label'            => 'Bank Transactions',
                'icon'             => null,
                'svg'              => null,
                'order'            => 62,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.bank.transactions.list',
                ],
                'show_condition' => $admin && bankActive(),
            ],
            'membership_plans' => [
                'parent_key'       => 'finance',
                'key'              => 'membership_plans',
                'route'            => 'dashboard.admin.finance.plans.index',
                'label'            => 'Membership Plans (old version)',
                'icon'             => null,
                'svg'              => null,
                'order'            => 63,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.plans.*',
                ],
                'show_condition' => false,
                'is_admin'       => true,
            ],
            'admin_finance_plan' => [
                'parent_key'       => 'finance',
                'key'              => 'admin_finance_plan',
                'route'            => 'dashboard.admin.finance.plan.index',
                'label'            => 'Pricing Plans',
                'icon'             => null,
                'svg'              => null,
                'order'            => 63,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.plan.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
                'badge'          => 'new',
            ],
            'payment_gateways' => [
                'parent_key'       => 'finance',
                'key'              => 'payment_gateways',
                'route'            => 'dashboard.admin.finance.paymentGateways.index',
                'label'            => 'Payment Gateways',
                'icon'             => null,
                'svg'              => null,
                'order'            => 64,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.paymentGateways.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'trial_features' => [
                'parent_key'       => 'finance',
                'key'              => 'trial_features',
                'route'            => 'dashboard.admin.finance.free.feature',
                'label'            => 'Trial Features',
                'icon'             => null,
                'svg'              => null,
                'order'            => 60,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.free.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'mobile_payment' => [
                'parent_key'       => 'finance',
                'key'              => 'mobile_payment',
                'route'            => 'dashboard.admin.finance.mobile.index',
                'label'            => 'Mobile Payment',
                'icon'             => null,
                'svg'              => null,
                'order'            => 60,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.mobile.index',
                ],
                'show_condition' => Helper::setting('mobile_payment_active', null, $setting),
                'is_admin'       => $admin,
            ],
            'pages' => [
                'parent_key'       => null,
                'key'              => 'pages',
                'route'            => 'dashboard.page.list',
                'label'            => 'Pages',
                'data-name'        => Introduction::ADMIN_PAGES,
                'icon'             => 'tabler-file-description',
                'svg'              => null,
                'order'            => 61,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.page.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'blog' => [
                'parent_key'       => null,
                'key'              => 'blog',
                'route'            => 'dashboard.blog.list',
                'label'            => 'Blog',
                'data-name'        => Introduction::ADMIN_BLOG,
                'icon'             => 'tabler-pencil',
                'svg'              => null,
                'order'            => 62,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.blog.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'affiliates_admin' => [
                'parent_key'       => null,
                'key'              => 'affiliates_admin',
                'route'            => 'dashboard.admin.affiliates.index',
                'label'            => 'Affiliates',
                'data-name'        => Introduction::ADMIN_AFFILIATES,
                'icon'             => 'tabler-currency-dollar',
                'svg'              => null,
                'order'            => 63,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.affiliates.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'coupons_admin' => [
                'parent_key'       => null,
                'key'              => 'coupons_admin',
                'route'            => 'dashboard.admin.coupons.index',
                'label'            => 'Coupons',
                'data-name'        => Introduction::ADMIN_COUPONS,
                'icon'             => 'tabler-ticket',
                'svg'              => null,
                'order'            => 64,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.coupons.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'email_templates' => [
                'parent_key'       => null,
                'key'              => 'email_templates',
                'route'            => 'dashboard.email-templates.index',
                'label'            => 'Email Templates',
                'data-name'        => Introduction::ADMIN_EMAIL_TEMPLATES,
                'icon'             => 'tabler-mail',
                'svg'              => null,
                'order'            => 65,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.email-templates.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'onboarding_pro_extension' => [
                'parent_key'       => null,
                'key'              => 'onboarding_pro_extension',
                'route'            => 'dashboard.admin.onboarding-pro.index',
                'label'            => 'Onboarding Pro',
                'data-name'        => Introduction::ADMIN_ONBOARDING_PRO,
                'icon'             => 'tabler-message-circle',
                'svg'              => null,
                'order'            => 65,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.onboarding-pro.*',
                ],
                'show_condition'   => Route::has('dashboard.admin.onboarding-pro.index'),
                'is_admin'         => true,
            ],
            'onboarding' => [
                'parent_key'       => null,
                'key'              => 'onboarding',
                'route'            => 'dashboard.admin.onboarding.index',
                'label'            => 'Onboarding',
                'data-name'        => Introduction::ADMIN_ONBOARDING,
                'icon'             => 'tabler-directions',
                'svg'              => null,
                'order'            => 65,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.onboarding.*',
                ],
                'show_condition' => Route::has('dashboard.admin.onboarding.index'),
                'is_admin'       => true,
            ],
            'mailchimp_newsletter' => [
                'parent_key'       => null,
                'key'              => 'mailchimp_newsletter',
                'route'            => 'dashboard.admin.mailchimp-newsletter.index',
                'label'            => 'Mailchimp Newsletter',
                'data-name'        => Introduction::ADMIN_MAILCHIMP_NEWSLETTER,
                'icon'             => 'tabler-mailbox',
                'svg'              => null,
                'order'            => 65,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.mailchimp-newsletter.*',
                ],
                'show_condition' => Route::has('dashboard.admin.mailchimp-newsletter.index'),
                'is_admin'       => true,
            ],
            'hubspot' => [
                'parent_key'       => null,
                'key'              => 'hubspot',
                'route'            => 'dashboard.admin.hubspot.index',
                'label'            => 'Hubspot',
                'data-name'        => Introduction::ADMIN_HUBSPOT_NEWSLETTER,
                'icon'             => 'tabler-affiliate',
                'svg'              => null,
                'order'            => 65,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.hubspot.*',
                ],
                'show_condition' => Route::has('dashboard.admin.hubspot.index'),
                'is_admin'       => true,
            ],
            'api_integration' => [
                'parent_key'       => null,
                'key'              => 'api_integration',
                'route'            => 'default',
                'label'            => 'API Integration',
                'data-name'        => Introduction::ADMIN_API_INTEGRATION,
                'icon'             => 'tabler-api',
                'svg'              => null,
                'order'            => 66,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.settings.openai', 'dashboard.admin.settings.gemini',
                    'dashboard.admin.settings.anthropic', 'dashboard.admin.settings.fal-ai',
                    'dashboard.admin.settings.stablediffusion', 'dashboard.admin.settings.unsplashapi',
                    'dashboard.admin.settings.pexelsapi', 'dashboard.admin.settings.pixabayapi',
                    'dashboard.admin.settings.serperapi',
                    'dashboard.admin.settings.tts', 'dashboard.admin.settings.synthesia',
                    'dashboard.admin.settings.aimlapi', 'dashboard.admin.settings.pebblely',
                    'dashboard.admin.settings.plagiarism', 'dashboard.admin.settings.clipdrop',
                    'dashboard.admin.settings.perplexity', 'dashboard.admin.settings.heygen',
                    'api_integration_deepseek', 'dashboard.admin.settings.ably',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'api_integration_azure_openai' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_azure_openai',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.azure-openai.index',
                'label'            => 'Azure OpenAI',
                'icon'             => null,
                'svg'              => null,
                'order'            => 67,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.azure-openai.*'],
                'show_condition'   => MarketplaceHelper::isRegistered('azure-openai'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
                'badge'            => 'new',
            ],
            'api_integration_openrouter' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_openrouter',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.open-router.show',
                'label'            => 'Open Router',
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.open-router.show'],
                'show_condition'   => Route::has('dashboard.admin.settings.open-router.show'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_ably' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_ably',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.ably',
                'label'            => 'Ably Setting',
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.open-router.show'],
                'show_condition'   => MarketplaceHelper::isRegistered('chatbot-agent'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
                'badge'            => 'new',
            ],
            'api_integration_llama' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_llama',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.llama',
                'label'            => 'Llama',
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.llama'],
                'show_condition'   => Route::has('dashboard.admin.settings.llama'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_searchapi' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_searchapi',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.searchapi',
                'label'            => 'Search Api',
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.searchapi'],
                'show_condition'   => Route::has('dashboard.admin.settings.searchapi'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_openai' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_openai',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.openai',
                'label'            => 'OpenAI',
                'icon'             => null,
                'svg'              => null,
                'order'            => 67,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.openai'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_gemini' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_gemini',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.gemini',
                'label'            => 'Gemini',
                'icon'             => null,
                'svg'              => null,
                'order'            => 67,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.gemini.*'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_anthropic' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_anthropic',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.anthropic',
                'label'            => 'Anthropic',
                'icon'             => null,
                'svg'              => null,
                'order'            => 68,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.anthropic'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
                'badge'            => 'new',
            ],
            'api_integration_deepseek' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_deepseek',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.deepseek',
                'label'            => 'Deepseek',
                'icon'             => null,
                'svg'              => null,
                'order'            => 68,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.deepseek'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
                'badge'            => 'new',
            ],
            'api_integration_fal_ai' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_fal_ai',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.fal-ai',
                'label'            => EngineEnum::FAL_AI->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 68,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.fal-ai'],
                'show_condition'   => Route::has('dashboard.admin.settings.fal-ai'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
                'badge'            => 'new',
            ],
            'api_integration_creatify' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_creatify',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.creatify',
                'label'            => EngineEnum::fromSlug('creatify')?->label() ?? 'Creatify',
                'icon'             => null,
                'svg'              => null,
                'order'            => 68,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.creatify'],
                'show_condition'   => Route::has('dashboard.admin.settings.creatify'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
                'badge'            => 'new',
            ],
            'api_integration_topview' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_topview',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.topview',
                'label'            => EngineEnum::TOPVIEW->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 68,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.topview'],
                'show_condition'   => Route::has('dashboard.admin.settings.topview'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
                'badge'            => 'new',
            ],
            'api_integration_vizard' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_vizard',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.vizard',
                'label'            => EngineEnum::VIZARD->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 68,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.vizard'],
                'show_condition'   => Route::has('dashboard.admin.settings.vizard'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
                'badge'            => 'new',
            ],
            'api_integration_klap' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_klap',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.klap',
                'label'            => EngineEnum::KLAP->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 68,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.klap'],
                'show_condition'   => Route::has('dashboard.admin.settings.klap'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
                'badge'            => 'new',
            ],
            'api_integration_piapi_ai' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_piapi_ai',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.piapi-ai',
                'label'            => trans('PiAPI'),
                'icon'             => null,
                'svg'              => null,
                'order'            => 69,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.piapi-ai'],
                'show_condition'   => Route::has('dashboard.admin.settings.piapi-ai'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
                'badge'            => 'new',
            ],
            'api_integration_stablediffusion' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_stablediffusion',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.stablediffusion',
                'label'            => 'StableDiffusion',
                'icon'             => null,
                'svg'              => null,
                'order'            => 69,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.stablediffusion'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_unsplashapi' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_unsplashapi',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.unsplashapi',
                'label'            => EngineEnum::UNSPLASH->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 70,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.unsplashapi'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_pexelsapi' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_pexelsapi',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.pexelsapi',
                'label'            => EngineEnum::PEXELS->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 71,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.pexelsapi'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_pixabayapi' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_pixabayapi',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.pixabayapi',
                'label'            => EngineEnum::PIXABAY->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 72,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.pixabayapi'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_serperapi' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_serperapi',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.serperapi',
                'label'            => EngineEnum::SERPER->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 73,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.serperapi'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_perplexity' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_perplexity',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.perplexity',
                'label'            => trans('Perplexity'),
                'icon'             => null,
                'svg'              => null,
                'order'            => 73,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.perplexity'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_tts' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_tts',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.tts',
                'label'            => 'TTS',
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.tts'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_synthesia' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_synthesia',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.synthesia',
                'label'            => __('Synthesia'),
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.synthesia'],
                'show_condition'   => Route::has('dashboard.user.ai-avatar.index'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_together' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_together',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.together',
                'label'            => __('Together'),
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.together'],
                'show_condition'   => MarketplaceHelper::isRegistered('ai-realtime-image'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_heygen' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_heygen',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.heygen',
                'label'            => __('Heygen'),
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.heygen'],
                'show_condition'   => Route::has('dashboard.user.ai-persona.index'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_music' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_aimlapi',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.aimlapi',
                'label'            => 'Aimlapi',
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.aimlapi'],
                'show_condition'   => Route::has('dashboard.user.ai-music.index'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_pebblely' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_pebblely',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.pebblely',
                'label'            => EngineEnum::PEBBLELY->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.pebblely'],
                'show_condition'   => Route::has('dashboard.user.ai-product-shot.index'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'plagiarism_extension' => [
                'parent_key'       => 'api_integration',
                'key'              => 'plagiarism_extension',
                'route'            => 'dashboard.admin.settings.plagiarism',
                'label'            => 'Plagiarism API',
                'icon'             => null,
                'svg'              => null,
                'order'            => 77,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.settings.plagiarism',
                ],
                'show_condition' => $admin && Route::has('dashboard.admin.settings.plagiarism'),
            ],
            'settings' => [
                'parent_key'       => null,
                'key'              => 'settings',
                'route'            => 'dashboard.admin.settings.general',
                'label'            => 'Settings',
                'data-name'        => Introduction::ADMIN_SETTINGS,
                'icon'             => 'tabler-device-laptop',
                'svg'              => null,
                'order'            => 75,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.settings.*', 'elseyyid.translations.home',
                    'elseyyid.translations.lang', 'dashboard.admin.config.*', 'dashboard.admin.ai-chat-model.index',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],

            'config' => [
                'parent_key'       => 'settings',
                'key'              => 'config',
                'route'            => 'dashboard.admin.config.index',
                'label'            => 'General Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 0,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.config.index'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'live_customizer' => [
                'parent_key'       => null,
                'key'              => 'live_customizer',
                'route'            => 'dashboard.admin.live-customizer.setting',
                'label'            => 'Live Customizer ',
                'icon'             => 'tabler-brush',
                'svg'              => null,
                'order'            => 69,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.live-customizer.setting'],
                'show_condition'   => MarketplaceHelper::isRegistered('live-customizer'),
                'is_admin'         => true,
            ],
            'thumbnail_system' => [
                'parent_key'       => 'settings',
                'key'              => 'thumbnail_system',
                'route'            => 'dashboard.admin.settings.thumbnail',
                'label'            => 'Thumbnail System',
                'icon'             => null,
                'svg'              => null,
                'order'            => 79,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.thumbnail'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'premium_advantages' => [
                'parent_key'       => 'settings',
                'key'              => 'premium_advantages',
                'route'            => 'dashboard.admin.config.premium-advantages.index',
                'label'            => 'Premium Advantages',
                'icon'             => null,
                'svg'              => null,
                'order'            => 79,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.config.premium-advantages.index'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'advanced_image_setting' => [
                'parent_key'       => 'settings',
                'key'              => 'advanced_image_setting',
                'route'            => 'dashboard.admin.settings.advanced-image.index',
                'label'            => 'AI Image Editor Setting',
                'icon'             => null,
                'svg'              => null,
                'order'            => 70,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.settings.advanced-image.index',
                ],
                'show_condition'   => MarketplaceHelper::isRegistered('advanced-image'),
                'is_admin'         => true,
            ],
            'privacy' => [
                'parent_key'       => 'settings',
                'key'              => 'privacy',
                'route'            => 'dashboard.admin.settings.privacy',
                'label'            => 'Privacy Policy and Terms',
                'icon'             => null,
                'svg'              => null,
                'order'            => 82,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.privacy'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'site_health' => [
                'parent_key'       => null,
                'key'              => 'site_health',
                'route'            => 'dashboard.admin.health.index',
                'label'            => 'Site Health',
                'data-name'        => Introduction::ADMIN_SITE_HEALTH,
                'icon'             => 'tabler-activity-heartbeat',
                'svg'              => null,
                'order'            => 85,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.health.index'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'license' => [
                'parent_key'       => null,
                'key'              => 'license',
                'route'            => 'dashboard.admin.license.index',
                'label'            => 'License',
                'data-name'        => Introduction::ADMIN_LICENSE,
                'icon'             => 'tabler-checklist',
                'svg'              => null,
                'order'            => 86,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.license.index'],
                'show_condition'   => Helper::appIsNotDemo(),
                'is_admin'         => true,
            ],
            'update' => [
                'parent_key'       => null,
                'class'            => 'nav-link--update',
                'key'              => 'update',
                'route'            => 'dashboard.admin.update.index',
                'label'            => 'Update',
                'data-name'        => Introduction::ADMIN_UPDATE,
                'icon'             => 'tabler-refresh',
                'svg'              => null,
                'order'            => 87,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.update.index'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'menu_setting' => [
                'parent_key'       => null,
                'class'            => '',
                'key'              => 'menu_setting',
                'route'            => 'dashboard.admin.menu.index',
                'label'            => 'Menu',
                'data-name'        => Introduction::ADMIN_MENU_SETTINGS,
                'icon'             => 'tabler-menu',
                'svg'              => null,
                'order'            => 88,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.menu.index'],
                'show_condition'   => Route::has('dashboard.admin.menu.index'),
                'is_admin'         => true,
            ],
            'footer_menu_setting' => [
                'parent_key'       => null,
                'class'            => '',
                'key'              => 'footer_menu_setting',
                'route'            => 'dashboard.admin.footer-menu.index',
                'label'            => 'Footer Menu',
                'data-name'        => Introduction::ADMIN_FOOTER_MENU_SETTINGS,
                'icon'             => 'tabler-menu',
                'svg'              => null,
                'order'            => 89,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.footer-menu.index'],
                'show_condition'   => Route::has('dashboard.admin.footer-menu.index') && setting('footer_menu_enabled', '0') == '1',
                'is_admin'         => true,
            ],
            'mega_menu_setting' => [
                'parent_key'       => null,
                'class'            => '',
                'key'              => 'mega_menu_setting',
                'route'            => 'dashboard.admin.mega-menu.index',
                'label'            => 'Mega Menu',
                'icon'             => 'tabler-menu-2',
                'svg'              => null,
                'order'            => 88,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.mega-menu.index'],
                'show_condition'   => Route::has('dashboard.admin.mega-menu.index'),
                'is_admin'         => true,
            ],
        ];

        return $this->mergeExtensionMenu($menu);
    }

    public function mergeExtensionMenu($menu)
    {
        // automation extension
        $is_automation_active = setting('ai_automation');

        $menu = array_merge($menu, [
            'openai_generator_extension' => [
                'parent_key'       => null,
                'key'              => 'openai_generator_extension',
                'route'            => 'default',
                'label'            => 'Ai Template',
                'icon'             => 'tabler-list-details',
                'svg'              => null,
                'order'            => 5,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.ai-template.openai-generator.*', 'dashboard.user.ai-template.openai-generator-filter.*',
                ],
                'show_condition' => Route::has('dashboard.user.ai-template.openai-generator.index') && Route::has('dashboard.user.ai-template.openai-generator-filter.index') && setting('ai_template_for_customer', '1') == '1',
            ],
            'custom_templates_extension' => [
                'parent_key'       => 'openai_generator_extension',
                'key'              => 'custom_templates_extension',
                'route'            => 'dashboard.user.ai-template.openai-generator.index',
                'label'            => 'Custom Templates',
                'icon'             => null,
                'svg'              => null,
                'order'            => 5,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.ai-template.openai-generator.*',
                ],
                'show_condition' => Route::has('dashboard.user.ai-template.openai-generator.index') && Route::has('dashboard.user.ai-template.openai-generator-filter.index') && setting('ai_template_for_customer', '1') == '1',
            ],
            'ai_writer_categories_extension' => [
                'parent_key'       => 'openai_generator_extension',
                'key'              => 'ai_writer_categories_extension',
                'route'            => 'dashboard.user.ai-template.openai-generator-filter.index',
                'label'            => 'AI Writer Categories',
                'icon'             => null,
                'svg'              => null,
                'order'            => 5,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.user.ai-template.openai-generator-filter.*'],
                'show_condition'   => Route::has('dashboard.user.ai-template.openai-generator.index') && Route::has('dashboard.user.ai-template.openai-generator-filter.index') && setting('ai_template_for_customer', '1') == '1',
            ],
            'ai_plagiarism_extension' => [
                'parent_key'       => null,
                'key'              => 'ai_plagiarism_extension',
                'route'            => 'dashboard.user.openai.plagiarism.index',
                'label'            => 'AI Plagiarism',
                'icon'             => 'tabler-progress-check',
                'svg'              => null,
                'order'            => 6,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.openai.plagiarism.index',
                ],
                'show_condition' => Route::has('dashboard.user.openai.plagiarism.index'),
            ],
            'ai_detector_extension' => [
                'parent_key'       => null,
                'key'              => 'ai_detector_extension',
                'route'            => 'dashboard.user.openai.detectaicontent.index',
                'label'            => 'AI Detector',
                'icon'             => 'tabler-text-scan-2',
                'svg'              => null,
                'order'            => 6,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.openai.detectaicontent.index',
                ],
                'show_condition' => Route::has('dashboard.user.openai.detectaicontent.index'),
            ],
            'ai_social_media_extension' => [
                'parent_key'       => null,
                'key'              => 'ai_social_media_extension',
                'route'            => 'dashboard.user.automation.index',
                'label'            => 'AI Social Media',
                'icon'             => 'tabler-share', // tabler-thumb-up
                'svg'              => null,
                'order'            => 6,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.automation.index',
                ],
                'show_condition' => Route::has('dashboard.user.automation.index') && $is_automation_active,
            ],
            'scheduled_posts_extension' => [
                'parent_key'       => null,
                'key'              => 'scheduled_posts_extension',
                'route'            => 'dashboard.user.automation.list',
                'label'            => 'Social Media Posts',
                'icon'             => 'tabler-report',
                'svg'              => null,
                'order'            => 6,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.automation.list',
                ],
                'show_condition' => Route::has('dashboard.user.automation.list') && $is_automation_active,
            ],
            'ai_social_media_settings_extension' => [
                'parent_key'       => 'settings',
                'key'              => 'ai_social_media_settings_extension',
                'route'            => 'dashboard.admin.automation.settings',
                'label'            => 'AI Social Media Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 77,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.automation.settings',
                ],
                'show_condition' => Route::has('dashboard.admin.automation.settings'),
            ],
            'ai_chat_pro_settings_extension' => [
                'parent_key'       => 'settings',
                'key'              => 'ai_chat_pro_settings_extension',
                'route'            => 'dashboard.admin.openai.chat.pro.settings',
                'label'            => 'AI Chat Pro Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 77,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.openai.chat.pro.settings',
                ],
                'show_condition' => Route::has('dashboard.admin.openai.chat.pro.settings'),
                'badge'          => 'new',
            ],
            'content_manager_extension' => [
                'parent_key'       => 'settings',
                'key'              => 'content_manager_extension',
                'route'            => 'content-manager::settings',
                'label'            => 'Content Manager Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 77,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'migration::settings.*',
                ],
                'show_condition' => MarketplaceHelper::isRegistered('content-manager'),
                'badge'          => 'new',
            ],
            'social_media_settings_extension' => [
                'parent_key'       => 'settings',
                'key'              => 'social_media_settings_extension',
                'route'            => 'dashboard.admin.social-media.setting.index',
                'label'            => 'Social Media Platform Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 78,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.social-media.setting.index',
                ],
                'show_condition' => Route::has('dashboard.admin.social-media.setting.index'),
            ],

            'chat_settings_extension' => [
                'parent_key'       => null,
                'key'              => 'chat_settings_extension',
                'route'            => 'default',
                'label'            => 'Chat Settings',
                'icon'             => 'tabler-message-circle',
                'svg'              => null,
                'order'            => 7,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.chat-setting.chat-category.*', 'dashboard.user.chat-setting.chatbot.*', 'dashboard.user.chat-setting.chat-template.*',
                ],
                'show_condition' => Route::has('dashboard.user.chat-setting.chat-category.index') && (setting('chat_setting_for_customer', 1) == 1),
            ],
            'chat_categories_extension' => [
                'parent_key'       => 'chat_settings_extension',
                'key'              => 'chat_categories_extension',
                'route'            => 'dashboard.user.chat-setting.chat-category.index',
                'label'            => 'Chat Categories',
                'icon'             => null,
                'svg'              => null,
                'order'            => 1,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.chat-setting.chat-category.*',
                ],
                'show_condition' => Route::has('dashboard.user.chat-setting.chat-category.index') && (setting('chat_setting_for_customer', 0) == 1),
            ],
            'chat_template_extension' => [
                'parent_key'       => 'chat_settings_extension',
                'key'              => 'chat_template_extension',
                'route'            => 'dashboard.user.chat-setting.chat-template.index',
                'label'            => 'Chat Templates',
                'icon'             => null,
                'svg'              => null,
                'order'            => 2,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.chat-setting.chat-template.*',
                ],
                'show_condition' => Route::has('dashboard.user.chat-setting.chat-template.index') && (setting('chat_setting_for_customer', '0') == 1),
            ],
            'chat_training_extension' => [
                'parent_key'       => 'chat_settings_extension',
                'key'              => 'chat_training_extension',
                'route'            => 'dashboard.user.chat-setting.chatbot.index',
                'label'            => 'Chatbot Training',
                'icon'             => null,
                'svg'              => null,
                'order'            => 3,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.chat-setting.chatbot.*',
                ],
                'show_condition' => Route::has('dashboard.user.chat-setting.chatbot.index') && (setting('chat_setting_for_customer', '0') == 1),
            ],
            'cloudflare_r2_extension' => [
                'parent_key'       => 'settings',
                'key'              => 'cloudflare_r2_extension',
                'route'            => 'dashboard.admin.settings.cloudflare-r2',
                'label'            => 'Cloudflare R2',
                'icon'             => null,
                'svg'              => null,
                'order'            => 8,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.settings.cloudflare-r2',
                ],
                'show_condition' => Route::has('dashboard.admin.settings.cloudflare-r2'),
            ],
            'photo_studio_extension' => [
                'parent_key'       => null,
                'key'              => 'photo_studio_extension',
                'route'            => 'dashboard.user.photo-studio.index',
                'label'            => 'AI Photo Studio',
                'icon'             => 'tabler-device-laptop',
                'svg'              => null,
                'order'            => 8,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.photo-studio.index',
                ],
                'show_condition' => Route::has('dashboard.user.photo-studio.index') && setting('photo_studio', 1) == 1,
            ],
            'seo_tool_extension' => [
                'parent_key'       => null,
                'key'              => 'seo_tool_extension',
                'route'            => 'dashboard.user.seo.index',
                'label'            => 'SEO Tool',
                'icon'             => 'tabler-seo',
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.seo.index',
                ],
                'show_condition' => Route::has('dashboard.user.seo.index') && setting('seo_ai_tool', 1) == 1,
            ],
            'ai_web_chat_extension' => [
                'parent_key'       => null,
                'key'              => 'ai_web_chat_extension',
                'route'            => 'dashboard.user.openai.webchat.workbook',
                'label'            => 'AI Web Chat',
                'icon'             => 'tabler-world-www',
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.openai.webchat.workbook',
                ],
                'show_condition' => Route::has('dashboard.user.openai.webchat.workbook'),
            ],
            'clipdrop_extension' => [
                'parent_key'       => 'api_integration',
                'key'              => 'clipdrop_extension',
                'route'            => 'dashboard.admin.settings.clipdrop',
                'label'            => 'Clipdrop',
                'icon'             => null,
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.settings.clipdrop',
                ],
                'show_condition' => Route::has('dashboard.admin.settings.clipdrop'),
            ],
            'novita_extension' => [
                'parent_key'       => 'api_integration',
                'key'              => 'novita_extension',
                'route'            => 'dashboard.admin.settings.novita',
                'label'            => 'Novita',
                'icon'             => null,
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.settings.novita',
                ],
                'show_condition' => Route::has('dashboard.admin.settings.novita'),
            ],
            'freepik_extension' => [
                'parent_key'       => 'api_integration',
                'key'              => 'freepik_extension',
                'route'            => 'dashboard.admin.settings.freepik',
                'label'            => 'Freepik',
                'icon'             => null,
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.settings.freepik',
                ],
                'show_condition' => Route::has('dashboard.admin.settings.freepik'),
                'badge'          => 'new',
            ],
            'x_ai' => [
                'parent_key'       => 'api_integration',
                'key'              => 'x_ai',
                'route'            => 'dashboard.admin.settings.x-ai',
                'label'            => 'X AI',
                'icon'             => null,
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.settings.x-ai',
                ],
                'show_condition' => true,
                'badge'          => 'new',
            ],
            'xero_extension' => [
                'parent_key'       => 'api_integration',
                'key'              => 'xero_extension',
                'route'            => 'dashboard.admin.settings.xero',
                'label'            => 'Xero API',
                'icon'             => null,
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.settings.xero',
                ],
                'show_condition' => MarketplaceHelper::isRegistered('xero'),
                'badge'          => 'new',
            ],
            'maintenance_setting' => [
                'parent_key'       => 'settings',
                'key'              => 'maintenance_setting',
                'route'            => 'dashboard.admin.settings.maintenance.index',
                'label'            => 'Maintenance',
                'icon'             => null,
                'svg'              => null,
                'order'            => 8,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'is_admin'         => true,
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.settings.maintenance.index',
                ],
                'show_condition' => Route::has('dashboard.admin.settings.maintenance.index'),
            ],
            'checkout_registration_extension' => [
                'parent_key'       => 'settings',
                'key'              => 'checkout_registration_extension',
                'route'            => 'dashboard.admin.checkout.registration.settings.index',
                'label'            => 'Checkout Registration',
                'icon'             => null,
                'svg'              => null,
                'order'            => 9,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.checkout.registration.settings.*',
                ],
                'show_condition' => MarketplaceHelper::isRegistered('checkout-registration'),
            ],
        ]);

        return $menu;
    }

    public function extensionCheck(string $slug): bool
    {
        return Extension::query()->where('slug', $slug)->where('installed', true)->exists();
    }

    public function showTeamFunctionality(): bool
    {
        $checkPlan = Plan::query()->where('is_team_plan', 1)->first();

        return Helper::setting('team_functionality') && ! auth()?->user()?->getAttribute('team_id') && $checkPlan;
    }

    /**
     * For new plan ai tools
     */
    public static function planAiToolsMenu(): array
    {
        $keys = [
            'ai_product_shot', 'ai_writer', 'ai_chat_all', 'ai_image_generator', 'ai_video', 'seo_tool_extension', 'ai_voiceover',
            'ai_pdf', 'ai_vision', 'ai_speech_to_text', 'photo_studio_extension', 'ai_rewriter', 'ai_editor',
            'ai_code_generator', 'ai_youtube', 'ai_chat_image', 'ai_rss', 'ai_voiceover_clone', 'ai_web_chat_extension',
            'ai_realtime_voice_chat', 'ai_social_media_extension', 'ai_detector_extension', 'ai_plagiarism_extension', 'ai_article_wizard', 'ai_voice_isolator', 'ext_chat_bot', 'ext_voice_chatbot', 'ext_social_media_dropdown',
        ];

        $data = (new self)->generate();

        foreach ($data as $datum) {
            $children = data_get($datum, 'children');

            if (is_array($children)) {

                $data = array_merge($data, $children);
            }
        }

        return collect($data)
            ->filter(function ($item) use ($keys) {
                if (! isset($item['show_condition'])) {
                    $item['show_condition'] = false;
                }

                return in_array($item['key'], $keys) && $item['show_condition'];
            })
            ->values()
            ->toArray();
    }

    public static function planFeatureMenu(): array
    {
        $data = (new self)->generate();

        foreach ($data as $datum) {
            $children = data_get($datum, 'children');

            if (is_array($children)) {

                $data = array_merge($data, $children);
            }
        }

        return collect($data)

            ->filter(function ($item) {
                if (! isset($item['show_condition'])) {
                    $item['show_condition'] = false;
                }

                return in_array($item['key'], [
                    'api_keys',
                    'brand_voice',
                    'support',
                    'integration',
                    'custom_templates_extension',
                    'chat_training_extension',
                    'creative_suite',
                    'ai_influencer',
                    'url_to_video',
                    'viral_clips',
                    'influencer_avatar',
                ]) && $item['show_condition'];
            })
            ->values()
            ->map(function ($item) {
                if ($item['key'] === 'api_keys') {
                    $item['label'] = 'Personal API Key';
                }
                if ($item['key'] === 'integration') {
                    $item['label'] = 'WordPress';
                }

                return $item;
            })
            ->toArray();
    }
}
