package com.saas.shared.dto;

import com.saas.shared.enums.Provider;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * Immutable DTO for VoIP configuration used by runtime controllers.
 * Decouples runtime layer from admin entities.
 * 
 * This DTO represents the resolved VoIP configuration for a tenant,
 * whether it comes from database or environment variables.
 */
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class VoipConfigDTO {
    
    /**
     * Tenant ID this config belongs to
     */
    private String tenantId;
    
    /**
     * VoIP provider (TELNYX, TWILIO, ZIWO)
     */
    private Provider provider;
    
    /**
     * AI Assistant ID for the voice AI
     * For Telnyx: Voice AI Assistant ID
     * For Twilio: TwiML App SID or similar
     */
    private String aiAssistantId;
    
    /**
     * Type of AI integration
     * Examples: "TELNYX_NATIVE_AI", "WEBSOCKET_STREAM", "TWILIO_OPENAI"
     */
    private String aiType;
    
    /**
     * Messaging profile ID for SMS
     */
    private String messagingProfileId;
    
    /**
     * WebSocket stream URL for AI processing
     * Used when aiType = "WEBSOCKET_STREAM"
     */
    private String streamUrl;
    
    /**
     * Indicates whether config is active
     */
    private boolean isActive;
    
    /**
     * Source of this configuration
     * true = from database, false = from environment variables
     */
    private boolean fromDatabase;
    
    /**
     * Check if this is a valid configuration
     */
    public boolean isValid() {
        if (provider == null) {
            return false;
        }
        
        // For TELNYX_NATIVE_AI, we need an AI assistant ID
        if ("TELNYX_NATIVE_AI".equals(aiType)) {
            return aiAssistantId != null && !aiAssistantId.trim().isEmpty();
        }
        
        // For WEBSOCKET_STREAM, we need a stream URL
        if ("WEBSOCKET_STREAM".equals(aiType)) {
            return streamUrl != null && !streamUrl.trim().isEmpty();
        }
        
        // Default: valid if provider is set
        return true;
    }
}
